#!/usr/bin/env python3
#
# Fit pairs of straight tracks
# Details in Exercise 1 of Homwork 7
#
# CC 18 Feb 2019
#--------------------------
import numpy as np
import math
import ccHistStuff as cc
import matplotlib.pyplot as plt
from scipy.optimize import curve_fit

# linear fit y = a*x + b
def myFunc(x, a, b):
    return a*x+b

# --------------------------------
# Here are the needed constants 
#---------------------------------
nDetectors = 4
w          = 0.005  # 50 micron is strip width
s          = np.full(nDetectors, w/np.math.sqrt(12))  # resolution in each hit
xdet       = np.array([2., 3., 5., 7.]) # x coordinates of detectors
#---------------------------

# read all data into numpy arrays
data = np.loadtxt("straightTracks.txt")
xv   = data[:,0]   # true xverteces  (called X0 in the exercise pdf)
yv   = data[:,1]   # true yverteces  (always zero, so useless)
nev  = len(xv)     # number of pairs of tracks
trk1 = data[:,[2,3,4,5]]   # hits for track number 1
trk2 = data[:,[6,7,8,9]]   # hits for track number 2

# reserve arrays for the fit results (y=ax+b)
a1 = np.full(nev, 0.)   # track 1  
b1 = np.full(nev, 0.)   # track 1
a2 = np.full(nev, 0.)   # track 2 
b2 = np.full(nev, 0.)   # track 2

# also reserve arrays for the uncertainty on the intercept
# (These are only needed for the optional part)
e1 = np.full(nev, 0.)   # track 1
e2 = np.full(nev, 0.)   # track 2

# Now we loop over pairs and we fit the pairs
# It would make sense to use np.polyfit
# However the covariance matrix from np.polyfit makes no sense
# https://mail.scipy.org/pipermail/numpy-discussion/2013-February/065649.html
# This has actually been fixed in a later version of np.ployfit but the
# version on the rpi is not fixed.
# So we will use scipy.optimize instead, sigh
for i in range(nev):  # loop over pairs
    y1 = trk1[i][:]   # coordinates of trk 1 for this pair
    y2 = trk2[i][:]   # coordinates of trk 2 for this pair
    # p1, cov1 = np.polyfit(xdet, y1, 1, w=1/s,  cov='unscaled')
    # p2, cov2 = np.polyfit(xdet, y2, 1, w=1/s,  cov='unscaled')
    p1, cov1 = curve_fit(myFunc, xdet, y1, p0=[0., 1.], sigma=s, absolute_sigma=True)
    p2, cov2 = curve_fit(myFunc, xdet, y2, p0=[0., 1.], sigma=s, absolute_sigma=True)
    a1[i] = p1[0]
    a2[i] = p2[0]
    b1[i] = p1[1]
    b2[i] = p2[1]
#-----------------------------------------------
# This is optional....the intercept is -b/a
# Propagation of errors
# sigma^2(b/a) =   sigma^2(b)/a^2
#                + sigma^2(a) (b^2/a^4)
#                - 2 * sigma(ab) (b/a^3)
#
# Note: if instead of fitting y=ax+b I had
# fitted y=c(x-d), then one of the fit parameters
# would have been the intercept ("d") and
# I could have just extracted its uncertainty
# from the diagonal element of the covariance matrix
#-----------------------------------------------------
    e1[i] = cov1[1,1]/(a1[i]*a1[i]) + cov1[0,0]*b1[i]*b1[i]/(a1[i]*a1[i]*a1[i]*a1[i])
    e1[i] = e1[i] - 2* cov1[0,1]*b1[i]/(a1[i]*a1[i]*a1[i])
    e2[i] = cov2[1,1]/(a2[i]*a2[i]) + cov2[0,0]*b2[i]*b2[i]/(a2[i]*a2[i]*a2[i]*a2[i])
    e2[i] = e2[i] - 2* cov2[0,1]*b2[i]/(a2[i]*a2[i]*a2[i])
    e1[i] = math.sqrt(e1[i])
    e2[i] = math.sqrt(e2[i])
    
# the intercepts with the y axis
x1   = -10000 * b1/a1   # in micron
x2   = -10000 * b2/a2  
xav  = 0.5 * (x1 + x2)  # the average 
dx   = np.concatenate( (x1-10000*xv, x2-10000*xv) )
dxav = xav - 10000*xv 

# These are only needed for the optional part
e1   =  10000 * e1   # in micron
e2   =  10000 * e2
e    = np.concatenate( (e1, e2) )
pull = dx/e
blah = 1./(e1*e1) + 1/(e2*e2)     # needed for weighted average below
dxw  = (x1/(e1*e1) + x2/(e2*e2))/blah - 10000*xv

# Plot x-xtrue for all tracks
fig, ax = plt.subplots(1,1)
b = np.linspace(-500,500,101)
con, bins, _ = ax.hist(dx, b, histtype='step', color='black')
ax.set_xlabel("Xi-X0 (microns)")
ax.set_xlim(bins[0], bins[-1])
cc.statBox(ax, dx, bins)
fig.show()
input("Press any key to continue")

# Plot xaverage - xtrue
fig2, ax2 = plt.subplots(1,1)
con2, bins2, _ = ax2.hist(dxav, b, histtype='step', color='black')
ax2.set_xlabel("Xav-X0 (microns)")
cc.statBox(ax2, dxav, bins2)
ax2.set_xlim(bins2[0], bins2[-1])
fig2.show()
input("Press <Enter> to continue") 

###########################################################
# What follows here are the bonus plots
###########################################################
# Plot the pull
fig3, ax3 = plt.subplots(1,1)
con3, bins3, _ = ax3.hist(pull, np.linspace(-3.5,3.5,71), histtype='step', color='black')
ax3.set_xlabel("(Xi-X0)/error (microns)")
cc.statBox(ax3, pull, bins3)
ax3.set_xlim(bins3[0], bins3[-1])
fig3.show()
input("Press <Enter> to continue") 

# Plot xaverage - xtrue (but now weighted average)
fig4, ax4 = plt.subplots(1,1)
con4, bins4, _ = ax4.hist(dxw, b, histtype='step', color='black')
ax4.set_xlabel("Xav_w-X0 (microns)")
cc.statBox(ax4, dxw, bins4)
ax4.set_xlim(bins4[0], bins4[-1])
fig4.show()
input("Press <Enter> to continue")

# The tracks with "shallowest" slope are the ones which
# will have th highest uncertainty on the intercept.
# At least, this is what I think...so scatter plot
# the error on the intercept vs. the absolute value
# of slope (yep, looks from the plot that I was right..)
fig5, ax5 = plt.subplots()
slope = np.concatenate( (a1, a2) )
ax5.plot(np.abs(slope), e, linestyle='none', marker='o', markersize=2)
ax5.set_ylim(0,1000)  
ax5.set_ylabel("Intercept uncertainty (microns)")
ax5.set_xlabel("Slope of track")
fig5.show()
input("Press <Enter> to continue")


