#------------------------------------------------------------------
# fitExp: Exponential fit to histogram contents
# fit2DTracksConstrained: constrained fit to two 2D tracks
#-------------------------------------------------------------------
import numpy as np
def fit2DTracksConstrained(x1, y1, s1, x2, y2, s2, guess, nIter=4, verbosity=0):
    """
    fits two 2D tracks as follows
    y1 = par[0]* (x1-par[2])
    y2 = par[1]* (x2-par[2])
    Inputs: 
      x1, y1, x2, y2 : measured coordinates
      s1, s2         : measurement errors (sigmas) on y1, y2
      guess          : initial guess to par
      nIter          : number of iterations
      verbosity      : >0 for some debug output
    Returns:
      par      : fitted parameters
      chisq    : chisquared at minimum
      ndof     : number of degrees of freedom
      cov      : covariance matrix
    """

    # we should check that len(x1)=len(y1)=len(s1) etc, 
    # but we shall be sloppy and assume that it is all fine
    
    # parameters
    par = guess.copy()

    # initial dy

    # number of points
    n1 = len(x1)  # 1st track
    n2 = len(x2)  # 2nd track
    n  = n1+n2    # total

    # concatenate array
    x  = np.concatenate( (x1,x2) )
    y  = np.concatenate( (y1,y2) )
    s  = np.concatenate( (s1,s2) )

    # build the covariance matrix of the measurements
    W = np.diag(1./(s*s))   # NxN matrix
    
    # Start the iteration
    for i in range(0, nIter):

        # fitted y coordinates
        y1fit = par[0] * (x1 - par[2])
        y2fit = par[1] * (x2 - par[2])
        yfit  = np.concatenate( (y1fit,y2fit) )

        # chisq
        chisq = ( (yfit-y)**2/(s*s) ).sum()
        if verbosity>0:
            print ("before iteration ", i, "chisq =", chisq)
            print( 10000* (y1fit-y1) )
            print( 10000* (y2fit-y2) )
        
        # derivatives..Could be done more compactly, but this is more readable
        dy1_dpar0 = x1 - par[2]
        dy1_dpar1 = 0. * y1
        dy1_dpar2 = np.full( (n1), -par[0] )
        dy2_dpar0 = 0. * y2
        dy2_dpar1 = x2 - par[2]
        dy2_dpar2 = np.full( (n2), -par[1] )

        dy_dpar0 = np.concatenate( (dy1_dpar0 , dy2_dpar0) )
        dy_dpar1 = np.concatenate( (dy1_dpar1 , dy2_dpar1) )
        dy_dpar2 = np.concatenate( (dy1_dpar2 , dy2_dpar2) )

        # matrices...A and Atrans are the derivatives of the predictions wrt the fitted parameters
        Atrans = np.array( [dy_dpar0, dy_dpar1, dy_dpar2] )   # 3 x N matrix
        A = (Atrans.T).copy()                                 # Nx3 matrix
        dy = (np.array( [(y-yfit),] )).T                      # Nx1 column vector

        # find the matrix to be inverted, and invert it
        temp  = np.matmul(Atrans, W)   # 3xN * NxN = 3xN
        temp2 = np.matmul(temp, A)     # 3xN * Nx3 = 3x3
        temp3 = np.linalg.inv(temp2)   # 3x3 ... this is the covariance matrix

        # multiply again
        temp4 = np.matmul(temp3, Atrans) # 3x3 * 3xN = 3xN
        temp5 = np.matmul(temp4, W)      # 3xN * NxN = 3xN
        dpar  = np.matmul(temp5, dy)     # 3xN * Nx1 = 3x1 column vector

        # the new values of the parameters
        par[0] = par[0] + dpar[0][0]
        par[1] = par[1] + dpar[1][0]
        par[2] = par[2] + dpar[2][0]

    # The fit is now done...calculate a few things
    ndof  =  n1 + n2 - len(par)
    y1fit = par[0] * (x1 - par[2])
    y2fit = par[1] * (x2 - par[2])
    yfit  = np.concatenate( (y1fit,y2fit) )
    chisq = ( (yfit-y)**2/ (s*s) ).sum()
    if verbosity>0:
        print ("At the end chisq =", chisq)
        print( 10000* (y1fit-y1) )
        print( 10000* (y2fit-y2) )
        
    # we are done
    return par, chisq, ndof, temp3
